/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 */

#include "serial_v2.h"
#include <QThread>
#include <QtSerialPort/QSerialPort>
#include <iostream>
#include <math.h>
#include <QDebug>
#include <QMessageBox>
#include <QApplication>
#include <QProcess>
#include "config_handler.h"

using namespace std;

QT_USE_NAMESPACE



/**********************************************************************************************************************************************************************************
 ** MISCELLANEOUS FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Setup the serial connection settings */
void serial_Setup(QSerialPort& serial, QString portname)
{
	serial.setPortName(portname);
	serial.setBaudRate(QSerialPort::Baud115200);
	serial.setDataBits(QSerialPort::Data8);
	serial.setParity(QSerialPort::NoParity);
	serial.setStopBits(QSerialPort::OneStop);
	serial.setFlowControl(QSerialPort::NoFlowControl);
	serial.setReadBufferSize(63);
}

/* Frequently used function for sending a line over serial and receiving a reply in one go.*/
QString serialWriteRead(QSerialPort& serial, QString tx, bool output_to_console)
{
	QString rx = "";
	int message_fail_counter = 0, message_fail_limit = 20;
	int reconnect_attempts = 0, reconnect_retries = 5;

	/* Check the read buffer for potential leftovers (for example reset related information data dump) */
	serialClearBuffer(serial);

	/* Write command string tx to serialport */
	serialWrite(serial, tx, output_to_console);

	/* Read response from signal generator */
	serialRead(serial, rx, message_fail_counter, message_fail_limit);

	/* If there has been a failure to receive a (complete) response, try resetting the serial connection */
	if (message_fail_counter >= message_fail_limit)
	{
		/* restarts application on external reset, but seems to knock out the console/application output... */
//		serial.close();
//		QMessageBox message;
//		message.critical(nullptr, "Packet Loss Detected!", "Cannot read data: packet loss detected.\n\nSoftware will attempt to restart.");
//		QProcess::startDetached(qApp->arguments()[0], qApp->arguments());
//		exit(-1);

		/* This chunk of code attempts to handle the connection when an event occurs that prevents serial communication (Reset, cable loose, etc.)
		 * It will reiniate the connection and then attempt to resend the last command given to the function.
		 * In case of a loose cable, this operation is flawed, because the SGx board has effectively been reset, but without the reset error, it does not follow the reset procedure! */
		QMessageBox message;
		message.critical(nullptr, "Packet Loss Detected!", "Cannot read data: packet loss detected.\n\nSoftware will attempt to reconnect.");
		qDebug() << "Packet loss detected!";

		for (reconnect_attempts = 1; reconnect_attempts <= reconnect_retries; reconnect_attempts++)
		{
			QList<QSerialPortInfo> temp_list = serial_get_SGX_port_quiet();
			if (!temp_list.isEmpty())
			{
				serial.setPortName(temp_list.at(0).portName());
				qDebug() << serial.portName();
				serial.close();
				QThread::msleep(1000);
				if (serial.open(QIODevice::ReadWrite))
				{
					break;
				}
			}
			QThread::msleep(1000);
		}

		/* If failed to reconnect with the SG port, give up */
		if ((reconnect_attempts >= reconnect_retries) && (message_fail_counter >= message_fail_limit))
		{
			QMessageBox message;
			message.critical(nullptr,"Message Lost", "Sent message been lost.\nSoftware state may be compromised.\n\nSoftware will shut down.");
			exit(-1);
		}


		/* After recovering from a failed communication attempt, retry sending the message */
		qDebug() << "resending the following command" << tx;
		output_to_console = true;
		message_fail_counter = 0;

		/* Check the read buffer for potential leftovers (for example reset related information data dump) */
		serialClearBuffer(serial);

		/* Write command string tx to serialport */
		serialWrite(serial, tx, output_to_console);

		/* Read response from signal generator */
		serialRead(serial, rx, message_fail_counter, message_fail_limit);

		/* If failed to reconnect with the SG port, give up */
		if (message_fail_counter >= message_fail_limit)
		{
			QMessageBox message;
			message.critical(nullptr,"Message Lost", "Sent message been lost.\nSoftware state may be compromised.\n\nSoftware will shut down.");
			exit(-1);
		}
	}

	/* Display the message response: */
	if (output_to_console == true)
	{
		qDebug() << "RX:\t" << rx << "\n";
	}

	return rx;
}


QList<QSerialPortInfo> serial_get_SGX_port_quiet()
{
	QList<QSerialPortInfo> infolist;
	ConfigHandler config(QCoreApplication::applicationDirPath() + "/config.txt");
	if (config.get_SGX_port(1) != "")
	{
		QSerialPortInfo info(config.get_SGX_port(1));
		if (!info.isNull())
		{
			infolist += info;
			qDebug() << info.portName();
		}
		if (infolist.isEmpty())
		{
			qDebug() << "Could not detect device on port" << config.get_SGX_port(1);
		}
	}
	else
	{
		for (const QSerialPortInfo& info : QSerialPortInfo::availablePorts())
		{
			if ((info.vendorIdentifier() == 8137 && info.productIdentifier() == 131))
			{
				infolist += info;
				qDebug() << info.portName();
			}
		}

		if (infolist.isEmpty())
		{
			qDebug() << "Could not auto-detect signal generator board";
		}
	}

	if (infolist.size() > 1)
	{
		qDebug() << "Multiple signal generator boards found";
	}
	return infolist;
}

/* Clear the serial buffer */
void serialClearBuffer(QSerialPort& serial)
{
	/* Check the read buffer for potential leftovers (for example reset related information data dump) */
	if (serial.bytesAvailable() > 0)
	{
		QString scavenge_RX;
		while(1)
		{
			if (serial.waitForReadyRead(500) == false)
			{
				scavenge_RX += QString::fromUtf8(serial.readAll());
				qDebug() << "BUFFER JUNK: " << scavenge_RX;
				break;
			}
			else
			{
				scavenge_RX += QString::fromUtf8(serial.readAll());
			}
		}
	}
}


/* Write a message to the serial buffer */
void serialWrite(QSerialPort& serial, QString tx, bool output_to_console)
{
	/* Add newline to outbound message so that signal generator will recognize the command */
	tx += "\n";

	/* Display the message to transmit if requested */
	if (output_to_console == true)
	{
		qDebug() << "TX:\t" << tx;
	}

	serial.write((const char*)(tx).toUtf8());
	serial.waitForBytesWritten(250);
}

/* Read from the serial buffer */
void serialRead(QSerialPort& serial, QString& rx, int& message_fail_counter, int& message_fail_limit)
{
	if (serial.waitForReadyRead(1000) || serial.bytesAvailable() > 0)
	{
		int bytes = serial.bytesAvailable();
		while (!rx.contains("\r\n"))
		{
			rx += serial.readAll();

			/* Extra handling for commands with long responses that fully fill up the read buffer.
			 * Long responses shouldn't count towards the message fail counter */
			while (bytes >= serial.readBufferSize() && !rx.contains("\r\n"))
			{
				//
				// TODO:
				// UART ONLY issue
				// Why does this waitForReadyRead ALWAYS timeout, regardless of configured time, while the one at the start of this function just returns true in a timely fashion.
				//

				serial.waitForReadyRead(1);
				bytes = serial.bytesAvailable();
				rx += serial.readAll();
			}

			message_fail_counter++;
//			qDebug() << "Reads:" << message_fail_counter << "/" << message_fail_limit << rx;

			if (message_fail_counter >= message_fail_limit)
			{
				return;
			}

			if (!rx.contains("\r\n"))
			{
				serial.waitForReadyRead(1);	//Workaround for UART.
			}
		}
	}
	else
	{
		/* Either the SGx is unreasonably slow to respond, or there has been no answer at all. */
		message_fail_counter = message_fail_limit;
	}
}


/* Make the board reset; clean up junk in the serial buffer */
QString serial_reset(QSerialPort& serial, QString channel)
{
	qDebug () << "Resetting SG board";
	QString return_val = serialWriteRead(serial, "$RST," + channel);
	while(1)
	{
		/* clear the buffer; wait 0.5 seconds; check if it's still empty. Keep clearing till it's all gone, then break out of the loop */
		serial.clear();
		if (serial.waitForReadyRead(500) == false)
		{
			break;
		}
	}
	//and then clear it one more time for good measure...
	serial.clear();
	qDebug () << "Serial buffer cleared after reset\n";
	return return_val;
}

/* Set the Channel */
QString serial_setChannel(QSerialPort& serial, QString channel)
{
	 return serialWriteRead (serial, "$CHANS," + channel);
}

/* Enable or disable the RF power. */
QString serial_setRF_enable(QSerialPort& serial, QString channel, bool input)
{
	qDebug() << "setRF enable:" << input;
	return serialWriteRead(serial, "$ECS," + channel + "," + QString::number((int)input));
}

/* Enable or disable the RF power. */
QString serial_getRF_status(QSerialPort& serial, QString channel, bool console_output_enabled)
{
	return serialWriteRead(serial, "$ECG," + channel, console_output_enabled);
}

/* enable or disable idle mode; prevents the SG Board from running hot all the time. */
QString serial_setIdle_enable(QSerialPort& serial, QString channel, bool input){
	qDebug() << "Idle mode is" << input;
	return serialWriteRead(serial, "$IDLES," + channel + "," + QString::number((int)input));
}

/* Set the PA type. Details are in the firmware */
QString serial_set_PA_type(QSerialPort& serial, QString channel, int PA_type)
{
	qDebug() << "PATS = " <<  channel + "," + QString::number(PA_type);
	return serialWriteRead(serial, "$PATS,"+ channel + "," + QString::number(PA_type));
}

/* Get the PA configuration. */
QString serial_get_PA_type(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$PATG,"+ channel);
}

/* Set the Clock Source. */
QString serial_set_clock_source(QSerialPort& serial, QString channel, int clock_source)
{
	qDebug() << "CSS = " <<  channel + "," + QString::number(clock_source);
	return serialWriteRead(serial, "$CSS," + channel + "," + QString::number(clock_source));
}

/* Get the Clock Source. */
QString serial_get_clock_source(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$CSG," + channel);
}

/* Get the board identity. */
QString serial_getIdentity(QSerialPort& serial, QString channel)
{
	qDebug() << "Requesting SG Board identity";
	QString rx = serialWriteRead(serial, "$IDN," + channel);
	return rx;
}

/* Get the error status of the board */
QString serial_getStatus(QSerialPort& serial, QString channel, bool console_output_enabled)
{
	if	(console_output_enabled){
		qDebug() << "Requesting SG Board status";
	}
	QString rx = serialWriteRead(serial, "$ST," + channel, console_output_enabled);
	return rx;
}

/* Clear the error status of the board */
QString serial_clearError(QSerialPort& serial, QString channel, bool console_output_enabled)
{
	return serialWriteRead(serial, "$ERRC," + channel, console_output_enabled);
}

/* Get the firmware version */
QString serial_getVersion(QSerialPort& serial, QString channel)
{
	qDebug() << "Requesting firmware version";
	return serialWriteRead(serial, "$VER," + channel);
}

/* returns the forward and reflected power in dBm. */
void serial_readPower_dbm(QSerialPort& serial, QString channel, double& frw, double& rfl, bool console_output_enabled)
{
	QString rx = serialWriteRead(serial, "$PPDG," + channel, console_output_enabled);
	rx = rx.remove("\r");
	rx = rx.remove("\n");
	QStringList get_list = rx.split(",");

	if (rx.contains("$PPDG,") && !rx.contains(",ERR"))
	{
		frw = get_list.at(2).toDouble();
		rfl = get_list.at(3).toDouble();

		if	(console_output_enabled)
		{
			qDebug() << "serial_readPower_dbm: " << frw << "\t" << rfl;
		}
	}
}

/* returns the forward and reflected power in watt. */
void serial_readPower_watt(QSerialPort& serial, QString channel, double& frw, double& rfl, bool console_output_enabled)
{
	 QString rx = serialWriteRead(serial, "$PPG," + channel, console_output_enabled);
	 rx = rx.remove("\r");
	 rx = rx.remove("\n");
	 QStringList get_list = rx.split(",");

	 if (rx.contains("$PPG,") && !rx.contains(",ERR"))
	 {
		 frw = get_list.at(2).toDouble();
		 rfl = get_list.at(3).toDouble();

		 if	(console_output_enabled)
		 {
			 qDebug() << "serial_readPower_watt: " << frw << "\t" << rfl;
		 }
	 }
}

/* Get the temperature of the setup; only works properly in certain PA setups! */
QString serial_getTemperature(QSerialPort& serial, QString channel, bool console_output_enabled)
{
	return serialWriteRead(serial, "$PTG," + channel, console_output_enabled);
}

/* Enable or disable Autogain.
 * This setting is normally not necessary, however $AIS disables autogain without re-enabling it when it's done */
QString serial_setAutoGain_enable(QSerialPort& serial, QString channel, bool input)
{
	qDebug() << "Autogain enable = " << input;
	return serialWriteRead(serial, "$AGES,"+ channel + "," + QString::number((int)input));
}

/* Enable or Disable analog input mode; Allows for 2.1V - 3.3V power control from the EXT_TRG pin
 * Note that input types change drastically depending on whether IQ DAC or VGA is used. IQ DAC uses percentage inputs for power, VGA uses dB's!
 * 8-bit unsigned, RF channel number (1-255), 0 for all channels
 * 8-bit unsigned, enable (1) / disable (0)
 * 8-bit unsigned, modulation select. (0=IQ DAC, 1=VGA)
 * 32-bit float, Power at minimum count;	% for IQ DAC	/	 dB for VGA
 * 16-bit unsigned, minimum count level
 * 32-bit float, Power at maximum count;	% for IQ DAC	/	 dB for VGA
 * 16-bit unsigned, maximum count level */
QString serial_setAnalogInput(QSerialPort& serial, QString channel, bool enable, int modulator_select, double power_min, int ADC_min, double power_max, double ADC_max)
{
	qDebug() << "$AIS," << channel << "," << enable << "," << modulator_select << "," << power_min << "," << ADC_min << "," << power_max << ADC_max;
	return serialWriteRead(serial, "$AIS," + channel + "," +
						   QString::number((int)enable) + "," +
						   QString::number(modulator_select) + "," +
						   QString::number(power_min) + "," +
						   QString::number(ADC_min) + "," +
						   QString::number(power_max) + "," +
						   QString::number(ADC_max));
}

/**********************************************************************************************************************************************************************************
 ** CW FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Set the frequency. input in MHz  */
QString serial_setFrequency(QSerialPort& serial, QString channel, double input_freq_MHz)
{
	qDebug() << "serial_setFrequency -> input value as string: " << QString::number(input_freq_MHz);
	return serialWriteRead(serial, "$FCS," + channel + "," + QString::number(input_freq_MHz));
}

/* Get the frequency in MHz */
QString serial_getFrequency(QSerialPort& serial, QString channel, bool console_output_enabled)
{
	return serialWriteRead(serial, "$FCG," + channel, console_output_enabled);
}

/* Set the output power setpoint in dBm */
QString serial_setPower_dbm(QSerialPort& serial, QString channel, double input_power_dbm)
{
	qDebug() << "serial_setPower_dbm -> input value as string: " << QString::number(input_power_dbm);
	return serialWriteRead(serial, "$PWRDS," + channel + "," + QString::number(input_power_dbm));
}

/* set output power setpoint in Watt */
QString serial_setPower_watt(QSerialPort& serial, QString channel, double input_power_watt)
{
	qDebug() << "serial_setPower_watt -> input value as string: " << QString::number(input_power_watt);
	return serialWriteRead(serial, "$PWRS," + channel + "," + QString::number(input_power_watt));
}

/* get the power setpoint in dbm */
QString serial_getPower_dbm(QSerialPort &serial, QString channel)
{
	return serialWriteRead(serial, "$PWRDG," + channel);
}

/* get the power setpoint in watt */
QString serial_getPower_watt(QSerialPort &serial, QString channel)
{
	return serialWriteRead(serial, "$PWRG," + channel);
}

/* Set the phase of the signal in degrees */
QString serial_setPhase(QSerialPort& serial, QString channel, double input_phase)
{
	qDebug() << "serial_setPhase -> input value as string: " << QString::number((int)input_phase);
	return serialWriteRead(serial, "$PCS," + channel + "," + QString::number((int)input_phase));
}

/* Get the phase of the signal in degrees */
QString serial_getPhase(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$PCG," + channel);
}

/* Set SGx board output power */
QString serial_setSGxPower_dbm(QSerialPort& serial, QString channel, double sgx_power_dbm)
{
	qDebug() << "serial_setSGxPower_dbm -> input value as string: " << QString::number(sgx_power_dbm);
	return serialWriteRead(serial, "$PWRSGDS," + channel + "," + QString::number(sgx_power_dbm));
}

/* Get SGx board output power */
QString serial_getSGxPower_dbm(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$PWRSGDG," + channel);
}

/* Set IQ modulator Magnitude */
QString serial_setMagnitude(QSerialPort& serial, QString channel, double magnitude)
{
	qDebug() << "serial_setMagnitude -> input value as string: " << QString::number((int)magnitude);
	return serialWriteRead(serial, "$MCS," + channel + "," + QString::number((int)magnitude));
}

/* Get IQ modulator Magnitude */
QString serial_getMagnitude(QSerialPort& serial, QString channel, bool console_output_enabled)
{
	return serialWriteRead(serial, "$MCG," + channel, console_output_enabled);
}

/* Set the VGA attentuation */
QString serial_setVGA_attenuation(QSerialPort& serial, QString channel, double attenuation_value)
{
	qDebug() << "GCS Attenuation: " << attenuation_value;
	return serialWriteRead(serial, "$GCS," + channel + "," + QString::number(attenuation_value,'f',1));
}

/* Get the VGA attentuation value */
QString serial_getVGA_attenuation(QSerialPort& serial, QString channel, bool console_output_enabled)
{
	return serialWriteRead(serial, "$GCG," + channel, console_output_enabled);
}

/**********************************************************************************************************************************************************************************
 ** PWM FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Enable PWM Mode. Function is kind of redundant as it technically does the same as setPWM_DC, but it's still a relatively clean ON/OFF so it is kept for ease of use. */
QString serial_setPWM_enable(QSerialPort& serial, QString channel, bool input, double input_percentage)
{
	qDebug() << "setPWM enable:" << input;
	if(input == true){
		return serialWriteRead(serial, "$DCS," + channel + "," + QString::number((int)input_percentage));
	}
	else{
		return serialWriteRead(serial, "$DCS," + channel + ",100");	//PWM = OFF
	}
}

/* Set the PWM duty cycle */
QString serial_setPWM_DC(QSerialPort& serial, QString channel, double input_percentage)
{
	qDebug() << "serial_setPWM_DC -> input value as string: " << QString::number((int)input_percentage);
	return serialWriteRead(serial, "$DCS," + channel + "," + QString::number((int)input_percentage));
}

/* Set the PWM frequency */
QString serial_setPWM_freq(QSerialPort& serial, QString channel, double input_freq_Hz, double correction_factor, double delay)
{
	//Note: value cannot be 0.
	qDebug() << "serial_setPWM_freq -> input value as string: " << QString::number((int)input_freq_Hz)+ "," + QString::number(correction_factor) + "," + QString::number(delay);
	return serialWriteRead(serial, "$DCFS," + channel + "," + QString::number((int)input_freq_Hz)+ "," + QString::number(correction_factor) + "," + QString::number(delay));
}

/* Set the PWM triggering:
 * mode: 1 = free running / 2 = triggered master / 3 = triggered slave
 * slave port: (optional) port for trigger pulse input	//slave mode
 * slave pin:  (optional) pin  for trigger pulse input	//slave mode
 * master port: (optional) port for trigger pulse output	//master mode
 * master pin:  (optional) pin  for trigger pulse output	//master mode
 *
 * Command can accept many more optional arguments on firmware level, but this function does not support it.
 * pulse config: (optional) configure the trigger pulse (bit 0 = invert or not, bit 15..8 = speed * 10usec)
 * refresh rate (optional) refresh rate in seconds */
QString serial_setPWM_triggering(QSerialPort& serial, QString channel, int PWM_trigger_mode, int slave_port, int slave_pin, int master_port, int master_pin, int trigger_pulse, int refresh_rate)
{
	//unused parameters - cast to void to prevent compiler warning.
	(void) slave_port, (void) slave_pin, (void) master_port, (void) master_pin, (void) trigger_pulse, (void) refresh_rate;

	qDebug() << "trigger mode = " << PWM_trigger_mode;
	switch(PWM_trigger_mode){
		case 1:
			return serialWriteRead(serial, "$DCTS," + channel + "," + QString::number(PWM_trigger_mode));
			break;
		case 2:
			qDebug() << "this functionality is not yet supported";
			//return serialWriteRead(serial, "$DCTS," + channel + "," + QString::number(PWM_trigger_mode) + "," + QString::number(slave_port) + "," + QString::number(slave_pin) + "," + QString::number(master_port) + "," + QString::number(master_pin));
			break;
		case 3:
			qDebug() << "this functionality is not yet supported";
			//return serialWriteRead(serial, "$DCTS," + channel + "," + QString::number(PWM_trigger_mode) + "," + QString::number(slave_port) + "," + QString::number(slave_pin) + "," + QString::number(master_port) + "," + QString::number(master_pin));
			break;
		default:
			//forced free running
			return serialWriteRead(serial, "$DCTS," + channel + "," + QString::number(PWM_trigger_mode));
			break;
	}
return "$DCTS,Function not supported yet.";
}

/* Get all PWM-related settings */
QString serial_getPWM_settings(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$DCG," + channel);
}

/**********************************************************************************************************************************************************************************
 ** DLL FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Enable DLL mode.
 * default output_level = 0  */
QString serial_setDLL_enable(QSerialPort& serial, QString channel, bool input)  //output level default is 0;
{
	qDebug() << "setDLL enable:" << input;
	return serialWriteRead(serial, "$DLES," + channel + "," + QString::number((int)input));		//update $CMD
}

/* Get DLL's enable state */
QString serial_getDLL_status(QSerialPort& serial, QString channel)
{
	qDebug() << "get DLL enable status";
	return serialWriteRead(serial, "$DLEG," + channel);
}

/* configure DLL settings
 * default DLL_samples = 1
 * default DLL_step_delay = 5*/
QString serial_setDLL_settings(QSerialPort& serial, QString channel, double DLL_freq_low, double DLL_freq_high, double DLL_start_freq,
							double DLL_step_freq, double DLL_threshold, int DLL_main_delay)
{
	qDebug() << "$DLCS," << channel << "," << DLL_freq_low << "," << DLL_freq_high << "," << DLL_start_freq << "," << DLL_step_freq << "," << DLL_threshold << "," << (int)DLL_main_delay;
	return serialWriteRead(serial, "$DLCS," + channel + "," +
						   QString::number(DLL_freq_low) + "," +
						   QString::number(DLL_freq_high) + "," +
						   QString::number(DLL_start_freq) + "," +
						   QString::number(DLL_step_freq) + "," +
						   QString::number(DLL_threshold) + "," +
						   QString::number((int)DLL_main_delay));
}

/* Get configured DLL settings */
QString serial_getDLL_settings(QSerialPort& serial, QString channel)
{
	qDebug() << "get DLL settings";
	return serialWriteRead(serial, "$DLCG," + channel);
}

/**********************************************************************************************************************************************************************************
 ** ALL FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Enables ALL mode. */
QString serial_setALL_enable(QSerialPort& serial, QString channel, bool input)
{
	qDebug() << "ALL Enabled = " << input;
	return serialWriteRead(serial, "$ALES," + channel + "," + QString::number((int)input));
}

/* Configure ALL parameters: lower frequency,  upper frequency, threshold. In Voltages */
QString serial_setALL_settings(QSerialPort& serial, QString channel, double lowerFreq, double upperFreq, double threshold)
{
	qDebug() <<  "ALL ALCS -> " << lowerFreq << "," << upperFreq << "," << threshold;
	return serialWriteRead(serial, "$ALCS," + channel + "," + QString::number(lowerFreq) + "," + QString::number(upperFreq) + "," + QString::number(threshold));
}

/* check if ALL is currently enabled */
QString serial_getALL_status(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$ALEG," + channel);
}

/* check ALL parameters: lower frequency,  upper frequency, threshold. In Voltages */
QString serial_getALL_settings(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$ALCG," + channel);
}

/* Set forward return path DVGA configuration */
QString serial_set_DVGA_forward(QSerialPort& serial, QString channel, bool input, double attenuation)
{
	qDebug() <<  "DVGA FWDS -> " << input << attenuation;
	return serialWriteRead(serial, "$FWDS," + channel + "," + QString::number((int)input) + "," + QString::number(attenuation));
}

/* Set reflected return path DVGA configuration */
QString serial_set_DVGA_reflected(QSerialPort& serial, QString channel, bool input, double attenuation)
{
	qDebug() <<  "DVGA RFLS -> " << input << attenuation;
	return serialWriteRead(serial, "$RFLS," + channel + "," + QString::number((int)input) + "," + QString::number(attenuation));
}

/* Get forward return path DVGA configuration */
QString serial_get_DVGA_forward(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$FWDG," + channel);
}

/* Get reflected return path DVGA configuration */
QString serial_get_DVGA_reflected(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$RFLG," + channel);
}

/**********************************************************************************************************************************************************************************
 ** RF / I2C SPLITTER FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Enable / Disable the I2C connectivity of PA channels behind an RF/I2C splitter board */
QString serial_setRF_splitter_channels_connect(QSerialPort& serial, QString channel, int enable_bitmask)
{
	qDebug() <<  "RF Splitter channels connect -> " << enable_bitmask;
	return serialWriteRead(serial, "$MCIES," + channel + "," + QString::number(enable_bitmask));
}

/**********************************************************************************************************************************************************************************
 ** PSU FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Set the PSU voltage setpoint */
QString serial_setPSU_voltage_setpoint(QSerialPort& serial, QString channel, double voltage_setpoint)
{
	qDebug() <<  "PSU voltage setpoint -> " << voltage_setpoint;
	return serialWriteRead(serial, "$PSUVS," + channel + "," + QString::number(voltage_setpoint));
}

/* Set the PSU current limit of all PSUs combined */
QString serial_setPSU_current_limit(QSerialPort& serial, QString channel, double current_limit)
{
	qDebug() <<  "PSU current limit -> " << current_limit;
	return serialWriteRead(serial, "$PSUIS," + channel + "," + QString::number(current_limit));
}

/* Set the PSU enable state */
QString serial_setPSU_enable(QSerialPort& serial, QString channel, bool input)
{
	qDebug() <<  "PSU enable -> " << input;
	return serialWriteRead(serial, "$PSUES," + channel + "," + QString::number((int)input));
}

/* Get the PSU enable state */
QString serial_getPSU_enable(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$PSUEG," + channel);
}

/* Get the PSU voltage setpoint */
QString serial_getPSU_voltage_setpoint(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$PSUVG," + channel);
}

/* Get the PSU current limit of all PSUs combined */
QString serial_getPSU_current_limit(QSerialPort& serial, QString channel)
{
	return serialWriteRead(serial, "$PSUIG," + channel);
}

/* Get individual PSU Voltage and Current of all PSU's in one go.
 * The order is PSU1 voltage, PSU1 current, PSU2 voltage, PSU2 current, etc. */
QString serial_getPSU_IU_All_Reading(QSerialPort& serial, QString channel, bool console_output_enabled)
{
	return serialWriteRead(serial, "$PSUDG," + channel, console_output_enabled);
}

/**********************************************************************************************************************************************************************************
 ** SWEEP FUNCTIONS
 *********************************************************************************************************************************************************************************/
/* Perform a sweep with Watt values as inputs and outputs */
/* Warning: This function has never been utilized and has not be updated */
QString serial_runSweep_watt(QSerialPort& serial, QString channel, double swpStartFrequency, double swpStopFrequency, double swpStepFrequency, double swpPower_watt)
{
	/* INPUT:  $SWP,[channel],[start frequency],[stop frequency],[step frequency],[delay ms],[power dBm],[averages],[output option]
	 * OUTPUT: $SWP, [actual frequency],[forward power],[reflected power]
	 * ...
	 * $SWP, OK */

	QString tx = "$SWP," + channel + "," + QString::number(swpStartFrequency) + "," + QString::number(swpStopFrequency) + "," +
			QString::number(swpStepFrequency) + "," + QString::number(swpPower_watt) + ",0";
	qDebug() << "Sweep: " << tx;
	serial.write((const char*)(tx+"\n").toUtf8());
	serial.waitForBytesWritten(250);

	#warning this function is neglected.
	QString rx;
	while(serial.waitForReadyRead(30000) || serial.bytesAvailable()){		//Time out in 30s if it receives no data / readyRead signal
		if (!rx.contains("$SWP,OK")){
			rx += QString::fromUtf8(serial.readAll());
		}
	}
	qDebug() << "RX:\t" << rx << "\n";
	return rx;
}

/* Perform a sweep with dBm values as inputs and outputs */
QString serial_runSweep_dbm(QSerialPort& serial, QString channel, double swpStartFrequency, double swpStopFrequency, double swpStepFrequency, double swpPower_dbm)
{
	//Input: $SWPD,[channel],[start frequency],[stop frequency],[step frequency],[delay ms],[power dBm],[averages],[output option]
	QString tx = "$SWPD," + channel + "," +
			QString::number(swpStartFrequency) + "," + QString::number(swpStopFrequency) + "," +
			QString::number(swpStepFrequency) + "," + QString::number(swpPower_dbm) + ",0";

	qDebug() << "TX:\t" << tx;
	serial.write((const char*)(tx+"\n").toUtf8());
	serial.waitForBytesWritten(250);

	//output: $SWPD, [actual frequency],[forward power],[reflected power] -----> $SWPD,OK
	QString rx;

	while(!(rx.contains("$SWPD") && rx.contains(",OK\r\n")))
	{
		if(serial.waitForReadyRead(30000) != true)
		{
			qDebug() << "Sweep: No end of message received.";
			QMessageBox message;
			message.warning(nullptr, "Communication Error", "No answer received from Sweep.");
			break;
		}
		else
		{
			rx += QString::fromUtf8(serial.readAll());
			if (rx.contains(",ERR")){
				break;
			}
		}
	}

	qDebug() << "RX:\t" << rx << "\n";
	return rx;
}


/**********************************************************************************************************************************************************************************
 * Transparent Mode
 * *******************************************************************************************************************************************************************************/
#ifndef ROTARY_ENCODER_ENABLED
/* Triggered by mainwindow when Remote Command Mode is enabled and a command is received on the input_UART (terminal).
 * Function puts the received data in a string and passes it along over the output_UART (SGx Board) */
void serial_RCM_Transmit(QIODevice& input_UART, QSerialPort& output_UART)
{
	/* close and reopen serial port when sending a message to ensure a $RST didn't break comms */
	QString input_rx = "";
	input_rx += QString::fromUtf8(input_UART.readAll());
	qDebug() << "inputRX: " << input_rx << "\n";

	if(output_UART.error() != QSerialPort::NoError)
	{
		qDebug() << output_UART.error() << output_UART.errorString();

		output_UART.close();
		if (!output_UART.open(QIODevice::ReadWrite))
		{
			QList<QSerialPortInfo> temp_list = serial_get_SGX_port_quiet();
			output_UART.setPortName(temp_list.at(0).portName());
			qDebug() << output_UART.portName();
			output_UART.open(QIODevice::ReadWrite);
		}
	}

	output_UART.write((const char*)(input_rx).toUtf8());
}

/* Triggered by mainwindow when Remote Command Mode is enabled and an answer is returned from the output_UART (Sgx Board).
 * Function puts the received data in a string and returns it back along the input_UART (terminal) */
void serial_RCM_Receive(QIODevice& input_UART, QSerialPort& output_UART)
{
	QString output_rx = "";
	output_rx += QString::fromUtf8(output_UART.readAll());
	qDebug() << "outputRX: " << output_rx << "\n";
	input_UART.write((const char*)(output_rx).toUtf8());
}


#if defined (Q_OS_LINUX)
void serial_RCM_Receive_RS485(QIODevice& input_UART, QSerialPort& output_UART, GPIOClass& gpio_flipper)
{
	/* Read SGx response from SGx port */
	QString output_rx = "";
	output_rx += QString::fromUtf8(output_UART.readAll());
	qDebug() << "outputRX: " << output_rx << "\n";

	/* RS-485 Half-Duplex Push to Talk enable */
	gpio_flipper.setValue("1");

	/* Start writing the the RCM port */
	input_UART.write((const char*)(output_rx).toUtf8());

	/* Calculate wait time and add a bit of extra 'safety' wait time before rounding */
	double wait_time = (((double)(output_rx.count() * 10) / output_UART.baudRate()) * 1000.0);
	wait_time = round(wait_time + 0.6);

	/* Ensure message is sent before closing the Push to Talk GPIO */
	int bytes_written = input_UART.bytesToWrite() - output_rx.count();

	int runs = 0;
	while ((bytes_written < output_rx.count()) || (input_UART.waitForBytesWritten((int)wait_time)))		//while there are any bytes to be written.
	{
		bytes_written += input_UART.bytesToWrite();
		runs++;
	}

	/* RS-485 Half-Duplex Push to Talk disable */
	gpio_flipper.setValue("0");

	//
	// TODO:
	// It's not nice that the push-to-talk GPIO is getting flipped every 63 bytes.
	// If possible it should only flip once per complete message. That ought to save some time in long messages...
	// P.S. a scope is necessary to properly debug this :(
	//
}
#endif

/**********************************************************************************************************************************************************************************
 * DEBUG
 * *******************************************************************************************************************************************************************************/
/* TEST function of the firmware. Functionality prone to change any time. */
QString serial_Test(QSerialPort& serial, QString channel, int count, bool console_output_enabled)
{
	return serialWriteRead(serial, "$TST," + channel + "," + QString::number(count), console_output_enabled);
}


#endif

#ifdef ROTARY_ENCODER_ENABLED
/**
 * @brief serial_RCM_Receive
 * @param input_UART
 * @param output_UART
 * Triggered by mainwindow if Transparent Mode is enabled and a command is received on the input_UART.
 * Function puts the received data in a string and passes it along over the output_UART
 * Also shares some of the data with mainwindow in order to update the GUI based on the contents of the remote commands.
 */
void serial_RCM_Receive(QSerialPort& input_UART, QSerialPort& output_UART,int& change_indicator, double& frequency, bool& rf_status, double& power_dbm, double& power_watt){
	 static QString input_rx = "";
	 QRegExp regexp;
	 regexp.setCaseSensitivity(Qt::CaseInsensitive);

	 if(input_UART.bytesAvailable() && !(input_rx.contains("\r") || input_rx.contains("\n"))){//||input_UART.waitForReadyRead(75)){
		  input_rx += QString::fromUtf8(input_UART.readAll());
	 }

	 if(input_rx.contains("\r") || input_rx.contains("\n")){
		  qDebug() << "inputRX: " << input_rx << "\n";
		  output_UART.write((const char*)(input_rx).toUtf8());

		  //Intercept commands relevant to the GUI
		  if (input_rx.contains("$ECS,", Qt::CaseInsensitive)){
				regexp.setPattern("\\$ECS,\\d+,(\\d)\\s*\r$");
				if (regexp.indexIn(input_rx)>=0) {
					 rf_status = (bool)(regexp.cap(1).toInt());
					 change_indicator = 1;
				}
		  }
		  else if (input_rx.contains("$FCS,", Qt::CaseInsensitive)){
				regexp.setPattern("\\$FCS,\\d+,(\\d+)(.?)(\\d+)\\s*\r$");
				if (regexp.indexIn(input_rx)>=0) {
					 QString temp = regexp.cap(1) + regexp.cap(2) + regexp.cap(3);
					 frequency = (temp.toDouble()) * 1000000;
					 change_indicator = 2;
				}
		  }
		  else if (input_rx.contains("$PWRD,", Qt::CaseInsensitive)){
				regexp.setPattern("\\$PWRD,\\d+,(\\d*)(.?)(\\d+)\\s*\r$");
				if (regexp.indexIn(input_rx)>=0) {
					 QString temp = regexp.cap(1) + regexp.cap(2) + regexp.cap(3);
					 power_dbm = temp.toDouble();
					 change_indicator = 3;
				}
		  }
		  else if (input_rx.contains("$PWR,", Qt::CaseInsensitive)){
				regexp.setPattern("\\$PWR,\\d+,(\\d*)(.?)(\\d+)\\s*\r$");
				if (regexp.indexIn(input_rx)>=0) {
					 QString temp = regexp.cap(1) + regexp.cap(2) + regexp.cap(3);
					 power_watt = temp.toDouble();
					 change_indicator = 4;
				}
		  }

		  input_rx = "";	  //reset the static QString
	 }
}

/**
 * @brief serial_RCM_Return
 * @param input_UART
 * @param output_UART
 * Triggered by mainwindow if Transparent Mode is enabled and an answer is returned from the output_UART.
 * Function puts the received data in a string and returns it back along the input_UART
 * Also shares some of the data with mainwindow in order to update the GUI based on the contents of the remote commands.
 */
void serial_RCM_Return(QSerialPort& input_UART, QSerialPort& output_UART, int& change_indicator, double& frw, double& rfl, double& temperature){
	 static QString output_rx = "";
	 QRegExp regexp;
	 regexp.setCaseSensitivity(Qt::CaseInsensitive);

	 if (output_UART.bytesAvailable() && !(output_rx.contains("\r") || output_rx.contains("\n"))){// || output_UART.waitForReadyRead(75)) {
		  output_rx += QString::fromUtf8(output_UART.readAll());
					 //output_rx += QString::fromUtf8(output_UART.readAll());
	 }

	 if(output_rx.contains("\r") || output_rx.contains("\n")){
		  qDebug() << "OUTPUT RX:" << output_rx << "\n";
		  input_UART.write((const char*)(output_rx).toUtf8());


		  //Intercept commands relevant to the GUI
		  if (output_rx.contains("$PAG,")){
			  regexp.setPattern("\\$PAG,1,(\\D?)(\\d+)\\D(\\d+),(\\D?)(\\d+)\\D(\\d+)");
			  if (regexp.indexIn(output_rx)>=0) {
					QString string1 = regexp.cap(1) + regexp.cap(2) + "." + regexp.cap(3);
					QString string2 = regexp.cap(4) + regexp.cap(5) + "." + regexp.cap(6);
					frw = string1.toDouble();
					rfl = string2.toDouble();
					change_indicator = 5;
			  }
		  }
		  else if (output_rx.contains("$CPARS,")){
			  regexp.setPattern("\\$CPARS,(\\d+),0x90,(\\d+),(\\d+),(\\d+),(\\d+),(\\d+),(\\d+)");
			  if (regexp.indexIn(output_rx)>=0){
					temperature = regexp.cap(7).toDouble();
					change_indicator = 6;
			  }
		  }
		  output_rx = "";	 //reset the static QString
	 }
}
#endif





